-- Builtins {{{
vim.loader.enable()

local disabled_built_ins = {
  "gzip",
  "zip",
  "zipPlugin",
  "tar",
  "tarPlugin",
  "getscript",
  "getscriptPlugin",
  "vimball",
  "vimballPlugin",
  "2html_plugin",
  "tutor_mode_plugin",
  "logipat",
  "rrhelper",
  "spellfile_plugin",
}

for _, plugin in pairs(disabled_built_ins) do
  vim.g["loaded_" .. plugin] = 1
end

if vim.fn.executable "/Users/chl/.local/lib/pynvim/bin/python3" == 1 then
  vim.g.loaded_python_provider = 0
  vim.g.python3_host_prog = "/Users/chl/.local/lib/pynvim/bin/python3"
end

vim.cmd [[set statusline=%!v:lua.Statusline()]]

-- vim.g.health = { style = "float" }
vim.g.clojure_maxlines = 50
vim.g.clojure_align_subforms = 1
vim.g.loaded_perl_provider = 0
vim.g.loaded_ruby_provider = 0
vim.g.loaded_node_provider = 0
vim.g.mapleader = ","
vim.g.maplocalleader = ","
vim.g.pumheight = 9
vim.g.netrw_banner = 0
vim.g.netrw_keepdir = 0
vim.g.netrw_localcopydircmd = "cp -r"
vim.g.netrw_home = "~/.local/state/nvim"
vim.g.python3_host_skip_check = 1
vim.g.tex_flavor = "latex"
vim.g.tex_conceal = "abdmgs"
vim.o.breakindent = true
vim.o.clipboard = "unnamedplus"
vim.o.completeopt = "menu,menuone,noselect,noinsert"
vim.o.ignorecase = true
vim.opt.diffopt = {
  "closeoff",
  "filler",
  "foldcolumn:2",
  "hiddenoff",
  "indent-heuristic",
  "internal",
  "algorithm:patience",
  "linematch:60"
}
vim.opt.foldenable = false
vim.opt.foldcolumn = "0"
-- vim.opt.foldtext = ""
vim.opt.foldlevel = 99
vim.opt.foldlevelstart = 0
vim.opt.path = { ".", "", ".." }
vim.opt.shortmess:append("I", "c")
vim.opt.wildignore = {
  "venv*/",
  "__pycache__/",
  ".pytest_cache/",
  "tags",
  "htmlcov/.coverage",
  "*.pyc",
  "package-lock.json",
  "yarn.lock",
}
vim.opt.listchars = {
  tab = "» ",
  nbsp = "¬",
  trail = "·",
  extends = "…",
  precedes = "‹",
}
vim.o.laststatus = 3
vim.o.mouse = "nv"
vim.o.smartcase = true
vim.o.splitbelow = true
vim.o.splitright = true
vim.o.undofile = true
vim.o.updatetime = 250
vim.wo.number = false
vim.wo.signcolumn = "auto"

if vim.fn.has "nvim-0.11" == 1 then
  vim.o.winborder = "single"
  vim.o.messagesopt = "hit-enter,history:1000"
  vim.o.tabclose = "uselast"
end

-- if vim.env.TMUX then
--   vim.loop.fs_write(2, "\27Ptmux;\27\27]11;?\7\27\\", -1, nil)
-- end

local fzf_theme = "bw"

vim.api.nvim_command([[
  augroup updatecolorscheme
  autocmd colorscheme * :hi normal guibg=NONE ctermbg=NONE
  augroup END
]])

vim.cmd.colorscheme "default"
-- vim.cmd.colorscheme "morequiet"

if vim.fn.executable "rg" == 1 then
  vim.o.grepprg = "rg --vimgrep --smart-case --no-heading --hidden"
  vim.o.grepformat = "%f:%l:%c:%m,%f:%l:%m"
end
-- }}}

-- Autocommands {{{
vim.api.nvim_exec2(
  [[
augroup gpg
  autocmd!
  autocmd BufReadPre,FileReadPre *.gpg setlocal shada= noswapfile noundofile nobackup
  autocmd BufRead *.gpg %!gpg -qd %
  autocmd BufWriteCmd *.gpg write !gpg --batch --yes -co %
augroup END
"augroup health
"  autocmd!
"  autocmd FileType checkhealth :set modifiable | silent! %s/\v( ?[^\x00-\x7F])//g
"augroup END
augroup term
  autocmd!
  autocmd TermOpen * tnoremap <Esc> <C-\><C-n>
  autocmd TermOpen * setlocal nonumber norelativenumber
augroup END
augroup fzf
  autocmd!
  autocmd FileType fzf tnoremap <buffer> <Esc> <Esc>
  autocmd FileType fzf set laststatus=0 noshowmode noruler
    \| autocmd BufLeave <buffer> set laststatus=3 showmode ruler
augroup END
augroup ftdetect
  autocmd!
  autocmd BufRead,BufNewFile *.swv set filetype=tex
  autocmd BufRead,BufNewFile *.m,*.nb,*.wl,*.wls set filetype=mma
  autocmd BufRead,BufNewFile *.sls,*.sps,*.ss set filetype=scheme
  autocmd BufNewFile,BufRead *.purs set filetype=purescript
  autocmd BufNewFile,BufRead *.lisp packadd! slimv
  autocmd BufNewFile,BufRead *.tex packadd! vimtex
augroup END
augroup whitespace
  autocmd!
  autocmd BufWritePre * %s/\s\+$//e
augroup END
]],
  {}
)

vim.api.nvim_create_autocmd({ "FileType" }, {
  pattern = {
    "netrw",
    "help",
    "man",
    "hoogle",
    "vimmacdictionary",
  },
  callback = function()
    vim.cmd [[
      nnoremap <silent> <buffer> <C-q> :close<CR>
      set nobuflisted
    ]]
  end,
})
-- }}}

-- Commands/Functions {{{
-- https://github.com/romainl/vim-cool/issues/9
vim.api.nvim_exec2(
  [[
noremap <expr> <Plug>(StopHL) execute('nohlsearch')[-1]
noremap! <expr> <Plug>(StopHL) execute('nohlsearch')[-1]

fu! HlSearch()
    let s:pos = match(getline('.'), @/, col('.') - 1) + 1
    if s:pos != col('.')
        call StopHL()
    endif
endfu

fu! StopHL()
    if !v:hlsearch || mode() isnot 'n'
        return
    else
        sil call feedkeys("\<Plug>(StopHL)", 'm')
    endif
endfu

augroup SearchHighlight
au!
    au CursorMoved * call HlSearch()
    au InsertEnter * call StopHL()
augroup end
]],
  {}
)

-- https://gist.github.com/romainl/eae0a260ab9c135390c30cd370c20cd7
vim.api.nvim_exec2(
  [[
function! Redir(cmd, rng, start, end)
	for win in range(1, winnr('$'))
		if getwinvar(win, 'scratch')
			execute win . 'windo close'
		endif
	endfor
	if a:cmd =~ '^!'
		let cmd = a:cmd =~' %'
			\ ? matchstr(substitute(a:cmd, ' %', ' ' . shellescape(escape(expand('%:p'), '\')), ''), '^!\zs.*')
			\ : matchstr(a:cmd, '^!\zs.*')
		if a:rng == 0
			let output = systemlist(cmd)
		else
			let joined_lines = join(getline(a:start, a:end), '\n')
			let cleaned_lines = substitute(shellescape(joined_lines), "'\\\\''", "\\\\'", 'g')
			let output = systemlist(cmd . " <<< $" . cleaned_lines)
		endif
	else
		redir => output
		execute a:cmd
		redir END
		let output = split(output, "\n")
	endif
	new
	let w:scratch = 1
	setlocal buftype=nofile bufhidden=wipe nobuflisted noswapfile
	call setline(1, output)
endfunction

" This command definition includes -bar, so that it is possible to "chain" Vim commands.
" Side effect: double quotes can't be used in external commands
command! -nargs=1 -complete=command -bar -range Redir silent call Redir(<q-args>, <range>, <line1>, <line2>)
command! -nargs=? Filter let @a='' | execute 'g/<args>/y A' | new | setlocal bt=nofile | put! a

command Timestamp let s:old_lang = v:lc_time
      \ | language time C
      \ | keeppatterns %substitute/Last [cC]hanged\?:\zs\(\s*\).*/\=submatch(1)..strftime("%Y-%m-%d")/e
      \ | execute 'language time' s:old_lang
      \ | unlet! s:old_lang
]],
  {}
)

-- https://www.kevinli.co/posts/2017-01-19-multiple-cursors-in-500-bytes-of-vimscript/
vim.api.nvim_exec2(
  [[
let g:mc = "y/\\V\<C-r>=escape(@\", '/')\<CR>\<CR>"
nnoremap cn *``cgn
nnoremap cN *``cgN
vnoremap <expr> cn g:mc . "``cgn"
vnoremap <expr> cN g:mc . "``cgN"
]],
  {}
)

function Diagstats()
  local diags = vim.diagnostic.count(0, { severity = { min = vim.diagnostic.severity.WARN } })
  local num_errors = diags[vim.diagnostic.severity.ERROR] or 0
  local num_warnings = diags[vim.diagnostic.severity.WARN] or 0
  local counts = num_errors + num_warnings
  local errno = ""
  if counts > 0 then
    errno = "!" .. tostring(counts)
  end
  return errno
end

local function format_lsp_progress(messages)
  local percentage
  local result = {}
  for _, msg in pairs(messages) do
    if msg.message then
      table.insert(result, msg.title .. ": " .. msg.message)
    else
      table.insert(result, msg.title)
    end
    if msg.percentage then
      percentage = math.max(percentage or 0, msg.percentage)
    end
  end
  if percentage then
    return string.format("%03d: %s", percentage, table.concat(result, ", "))
  else
    return table.concat(result, ", ")
  end
end

local function format_uri(uri)
  if vim.startswith(uri, "jdt://") then
    local package = uri:match "contents/[%a%d._-]+/([%a%d._-]+)" or ""
    local class = uri:match "contents/[%a%d._-]+/[%a%d._-]+/([%a%d$]+).class" or ""
    return string.format("%s::%s", package, class)
  else
    return vim.fn.fnamemodify(vim.uri_to_fname(uri), ":.")
  end
end

-- Mathias Fußenegger, https://is.gd/BQRt7j
function FileLsp()
  local mode = vim.api.nvim_get_mode().mode
  if vim.lsp.status then
    local lsp_status = vim.lsp.status()
    if mode ~= "n" or lsp_status == "" then
      return format_uri(vim.uri_from_bufnr(vim.api.nvim_get_current_buf()))
    end
    return lsp_status
  end

  local messages = vim.lsp.util.get_progress_messages()
  if mode ~= "n" or vim.tbl_isempty(messages) then
    return format_uri(vim.uri_from_bufnr(vim.api.nvim_get_current_buf()))
  end
  return format_lsp_progress(messages)
end

function Statusline()
  local parts = {
    [[ %{luaeval("FileLsp()")} %m%r %= ]],
    "%#warningmsg#",
    "%{&ff!='unix'?'['.&ff.']':''}",
    "%*",
    "%#warningmsg#",
    "%{(&fenc!='utf-8'&&&fenc!='')?'['.&fenc.']':''}",
    "%*",
    "%{&paste?'¶':''} ",
    [[ %{luaeval("Diagstats()")} ]],
    "%{'≡ '.&tw}%{&et?'':'  »'} ",
    "%{&spell?'['.&spelllang.']':''} ",
    "%l %P ",
  }
  return table.concat(parts, "")
end

function Only()
  local cur_buf = vim.api.nvim_get_current_buf()
  for _, buf in ipairs(vim.api.nvim_list_bufs()) do
    if cur_buf ~= buf then
      pcall(vim.cmd.bd, buf)
    end
  end
end

function ToggleQuickFix()
  if vim.fn.empty(vim.fn.filter(vim.fn.getwininfo(), "v:val.quickfix")) == 1 then
    vim.cmd [[copen]]
  else
    vim.cmd [[cclose]]
  end
end

vim.api.nvim_exec2(
  [[
" https://phelipetls.github.io/posts/code-formatting-vim/
" https://github.com/phelipetls/dotfiles/blob/master/.config/nvim/autoload/format.vim
function! s:Format(...)
  silent keepjumps normal! '[v']gq
  if v:shell_error > 0
    silent undo
    echohl ErrorMsg
    echomsg 'formatprg "' . &formatprg . '" exited with status ' . v:shell_error
    echohl None
  endif
endfunction

function! FormatFile() abort
  let w:view = winsaveview()
  keepjumps normal! gg
  set operatorfunc=<SID>Format
  keepjumps normal! g@G
  keepjumps call winrestview(w:view)
  unlet w:view
endfunction

command! Format call FormatFile()
]],
  {}
)

vim.api.nvim_exec2(
  [[
"" Credit: https://github.com/strager/dotfiles
"" http://vim.wikia.com/wiki/Regex-based_text_alignment
function! s:align_section(regex) range abort
  let extra = 1
  let sep = empty(a:regex) ? '=' : a:regex
  let maxpos = 0
  let section = getline(a:firstline, a:lastline)
  for line in section
    let pos = match(line, ' *'.sep)
    if maxpos < pos
      let maxpos = pos
    endif
  endfor
  call map(section, 's:align_line(v:val, sep, maxpos, extra)')
  call setline(a:firstline, section)
endfunction

function! s:align_line(line, sep, maxpos, extra) abort
  let m = matchlist(a:line, '\(.\{-}\) \{-}\('.a:sep.'.*\)')
  if empty(m)
    return a:line
  endif
  let spaces = repeat(' ', a:maxpos - strlen(m[1]) + a:extra)
  return m[1] . spaces . m[2]
endfunction

command! -nargs=? -range Align <line1>,<line2>call s:align_section('<args>')
]],
  {}
)

vim.cmd [[command! -bar -nargs=* Jump cexpr system('git jump ' . expand(<q-args>))]]
vim.cmd [[command! -range GB echo join(systemlist("git -C " . shellescape(expand('%:p:h')) . " blame -L <line1>,<line2> " . expand('%:t')), "\n")]]
vim.cmd [[command! -complete=dir -nargs=+ Grep silent grep <args> | redraw!]]
vim.cmd [[command! -range Gist call system('gist', getbufline('%', <line1>, <line2>))]]
vim.cmd [[command! Changelog execute "normal i<C-R>=strftime('%Y-%m-%d')<CR><ESC>$a chl <chl@aliquote.org> -"]]

vim.api.nvim_create_user_command(
  "LspStop",
  function(kwargs)
    local name = kwargs.fargs[1]
    for _, client in ipairs(vim.lsp.get_clients({ name = name })) do
      client:stop()
    end
  end,
  {
    nargs = "?",
    complete = function()
      return vim.tbl_map(function(c) return c.name end, vim.lsp.get_clients())
    end
  }
)

vim.api.nvim_create_user_command(
  "LspRestart",
  function(kwargs)
    local name = kwargs.fargs[1]
    for _, client in ipairs(vim.lsp.get_clients({ name = name })) do
      local bufs = vim.lsp.get_buffers_by_client_id(client.id)
      client:stop()
      vim.wait(30000, function()
        return vim.lsp.get_client_by_id(client.id) == nil
      end)
      local client_id = vim.lsp.start(client.config, { attach = false })
      if client_id then
        for _, buf in ipairs(bufs) do
          vim.lsp.buf_attach_client(buf, client_id)
        end
      end
    end
  end,
  {
    nargs = "?",
    complete = function()
      return vim.tbl_map(function(c) return c.name end, vim.lsp.get_clients())
    end
  }
)
-- }}}

-- Mapping's {{{
vim.keymap.set("c", "<C-a>", [[<home>]], {})
vim.keymap.set("c", "<C-e>", [[<end>]], {})
vim.keymap.set("i", "<C-s>", [[<c-g>u<Esc>[s1z=`]a<c-g>u]])
vim.keymap.set("i", "jk", "<Esc>")
vim.keymap.set("t", "jk", [[<C-\><C-n>]])
vim.keymap.set("v", "/", '"fy/\\V<C-R>f<CR>')
vim.keymap.set("n", "-", [[<cmd>Ex<cr>]], {})
vim.keymap.set("n", "<leader>n", [[:set invrelativenumber]], {})
vim.keymap.set("n", "<C-q>", ":lua ToggleQuickFix()<cr>", { silent = true })
vim.keymap.set("n", "<C-s>", [[:setlocal spell!<bar>setlocal spell?<cr>:setlocal spelllang=en<cr>]])
vim.keymap.set("n", "<leader>x", ":lua Only()<cr>", { silent = true })
vim.keymap.set("n", "<leader>.", [[<cmd>lcd %:p:h<cr>]], {})
vim.keymap.set("n", "<leader><leader>", [[<C-^>]], {})
vim.keymap.set("n", "<leader><tab>", [[<cmd>$tabnew<cr>]], {})
vim.keymap.set("n", "<leader>s", [[vip:sort u<cr>]])
vim.keymap.set("v", "<leader>s", [[:sort u<cr>]])
vim.keymap.set("n", "<leader>S", [[vip:sort iu<cr>]])
vim.keymap.set("v", "<leader>S", [[:sort iu<cr>]])
vim.keymap.set("n", "g=", "<cmd>Format<cr>")
vim.keymap.set("n", "gb", "`[v`]")
vim.keymap.set("n", "J", "mzJ`z")
vim.keymap.set("n", "N", "Nzz")
vim.keymap.set("n", "/", "ms/")
vim.keymap.set("n", "U", "<C-r>")
vim.keymap.set("n", "j", "v:count == 0 ? 'gj' : 'j'", { expr = true, silent = true })
vim.keymap.set("n", "k", "v:count == 0 ? 'gk' : 'k'", { expr = true, silent = true })
vim.keymap.set("n", "n", "nzz")
vim.keymap.set("n", "zs", "1z=")
vim.keymap.set("v", "_`", [["zdi`<C-R>z`]])
vim.keymap.set("v", '_"', [["zdi"<C-R>z"]])
vim.keymap.set("v", "_(", [["zdi(<C-R>z)]])
vim.keymap.set("v", "_[", '"zdi[<C-R>z]')
vim.keymap.set("n", "(", "[", { remap = true })
vim.keymap.set("n", ")", "]", { remap = true })
vim.keymap.set("o", "(", "[", { remap = true })
vim.keymap.set("o", ")", "]", { remap = true })
vim.keymap.set("x", "(", "[", { remap = true })
vim.keymap.set("x", ")", "]", { remap = true })
vim.keymap.set("n", "((", "[[", { remap = true })
vim.keymap.set("n", "))", "]]", { remap = true })

vim.keymap.set("n", "<leader>:", require("fzf-lua").resume, {})
vim.keymap.set("n", "<leader>r", require("fzf-lua").oldfiles, {})
vim.keymap.set("n", "<C-f>", function()
  vim.fn.setreg("w", vim.fn.expand "<cword>")
  require("fzf-lua").live_grep {
    winopts = {
      on_create = function()
        local b = vim.api.nvim_get_current_buf()
        vim.keymap.set("t", "<C-r>", [['<C-\><C-N>"'.nr2char(getchar()).'pi']], { buffer = b, expr = true })
      end,
    },
  }
end, { silent = true })
vim.keymap.set("n", "<leader>?", function()
  require("fzf-lua").builtin { winopts = { width = 1, height = 0.2, row = 1 } }
end, { silent = true })
vim.keymap.set("n", "<leader>=", require("fzf-lua").tmux_buffers, {})
vim.keymap.set("n", "<C-p>", require("fzf-lua").files, {})
vim.keymap.set("n", "<leader>;", require("fzf-lua").buffers, {})
vim.keymap.set("n", "<leader>/", require("fzf-lua").blines, {})
vim.keymap.set("n", "<leader>WW", require("fzf-lua").diagnostics_workspace, {})
vim.keymap.set("n", "<leader>ww", require("fzf-lua").diagnostics_document, {})
vim.keymap.set("n", "<leader>gb", function()
  require("fzf-lua").git_bcommits { winopts = { preview = { hidden = "nohidden" } } }
end, { silent = true })
vim.keymap.set("n", "<leader>gg", require("fzf-lua").git_status, {})
-- }}}

-- Plugins {{{
-- Treesitter {{{2
require("nvim-treesitter.configs").setup {
  ensure_installed = {
    "bash",
    "bibtex",
    "c",
    "clojure",
    "comment",
    "commonlisp",
    "cpp",
    "elm",
    "go",
    "haskell",
    "javascript",
    "julia",
    "latex",
    "lua",
    "markdown",
    "markdown_inline",
    "org",
    "python",
    "r",
    "racket",
    "regex",
    "rust",
    "scheme",
    "swift",
    "tsx",
    "typescript",
    "vim",
    "vimdoc",
  },
  auto_install = false,
  highlight = { enable = true, disable = { "latex" } },
  indent = { enable = true, disable = { "latex" } },
  incremental_selection = {
    enable = true,
    keymaps = {
      init_selection = "<tab><tab>",
      node_incremental = "<tab>",
      node_decremental = "<S-tab>",
    },
  },
  textobjects = {
    select = {
      enable = true,
      lookahead = true,
      keymaps = {
        ["aa"] = "@parameter.outer",
        ["ia"] = "@parameter.inner",
        ["af"] = "@function.outer",
        ["if"] = "@function.inner",
        ["ac"] = "@class.outer",
        ["ic"] = "@class.inner",
        ["ab"] = "@block.outer",
        ["ib"] = "@block.inner",
      },
      selection_modes = {
        ["@parameter.outer"] = "v",
        ["@function.outer"] = "V",
        ["@class.outer"] = "<c-v>",
      },
    },
  },
}
---@class wolfram
local parser_config = require("nvim-treesitter.parsers").get_parser_configs()
parser_config.wolfram = {
  install_info = {
    url = "https://github.com/LumaKernel/tree-sitter-wolfram",
    files = { "src/parser.c", "src/scanner.c" },
    branch = "main",
    generate_requires_npm = true,
    requires_generate_from_grammar = true,
  },
  filetype = "mma",
}
vim.treesitter.language.register("wolfram", "mma")
-- }}}

-- LSP {{{2
-- NOTE: omnifunc autocompletion is sometimes managed in ftplugin/*.lua
vim.api.nvim_create_autocmd("LspAttach", {
  callback = function(args)
    local client = vim.lsp.get_client_by_id(args.data.client_id)
    local augroup = vim.api.nvim_create_augroup("LspFormatting", {})
    vim.keymap.set("i", "<C-k>", vim.lsp.buf.signature_help, { buffer = args.buf })
    vim.keymap.set("n", "K", vim.lsp.buf.hover, { buffer = args.buf })
    if client and client.server_capabilities.semanticTokensProvider then
      client.server_capabilities.semanticTokensProvider = nil
    end
    if client and client.server_capabilities.completionProvider then
      vim.lsp.completion.enable(true, client.id, args.buf, { autotrigger = false })
    end
    if client and client.name == "ruff_lsp" then
      client.server_capabilities.hoverProvider = false
    end
    -- NOTE: check if this is useful
    if client and client.name == "texlab" then
      vim.schedule(function()
        if vim.lsp.inlay_hint then
          vim.lsp.inlay_hint(args.buf, false)
        end
      end)
    end
    vim.keymap.set("n", "gd",
      "<cmd>lua require('fzf-lua').lsp_definitions({ jump1 = true, ignore_current_line=true })<cr>")
    vim.keymap.set("n", "gD", "<cmd>lua require('fzf-lua').lsp_declarations()<cr>")
    vim.keymap.set("n", "gr?", "<cmd>lua require('fzf-lua').lsp_finder()<cr>")
    vim.keymap.set("n", "gr",
      "<cmd>lua require('fzf-lua').lsp_references({ ignore_current_line = true, includeDeclaration = false })<cr>")
    if client and client.server_capabilities.inlayHintProvider then
      vim.keymap.set("n", "zI", "<cmd>lua vim.lsp.inlay_hint.enable(not vim.lsp.inlay_hint.is_enabled())<cr>", {})
    end
    if client and client.server_capabilities.documentSymbolProvider then
      vim.keymap.set("n", "gO", "<cmd>lua require('fzf-lua').lsp_document_symbols()<cr>")
    elseif client and client.server_capabilities.workspaceSymbolProvider then
      vim.keymap.set("n", "gO", "<cmd>lua require('fzf-lua').lsp_workspace_symbols()<cr>")
    end
    if client and client.server_capabilities.codeLensProvider then
      vim.api.nvim_create_autocmd({ "CursorMoved" }, {
        callback = function()
          vim.lsp.codelens.refresh()
        end,
      })
      vim.keymap.set("n", "z!", vim.lsp.codelens.run, { buffer = args.buf, silent = true })
    end
    if client and client.server_capabilities.documentFormattingProvider then
      vim.keymap.set({ "n", "x" }, "g=", "<cmd>lua vim.lsp.buf.format({async = true})<cr>", { buffer = args.buf })
      vim.api.nvim_clear_autocmds { group = augroup, buffer = args.buf }
      vim.api.nvim_create_autocmd({ "BufWritePre" }, {
        group = augroup,
        buffer = args.buf,
        callback = function()
          vim.lsp.buf.format { async = false, timeout_ms = 10000 }
        end,
      })
    end
  end,
})

vim.lsp.enable { "clangd", "r_language_server", "jdtls", "lua_ls", "lsp_wl", "bashls", "rust-analyzer", "hls", "racket-langserver", "clojure-lsp", "sourcekit" }

vim.diagnostic.config {
  virtual_text = false,
  float = {
    format = function(diagnostic)
      if diagnostic.source == "rustc" and diagnostic.user_data.lsp.data ~= nil then
        return diagnostic.user_data.lsp.data.rendered
      else
        return diagnostic.message
      end
    end,
  },
  signs = false,
  underline = true,
  update_in_insert = false,
  severity_sort = false,
}
-- }}}

-- fzf-lua {{{2
local actions = require "fzf-lua.actions"
require("fzf-lua").setup {
  "max-perf",
  -- fzf_colors = true,
  diagnostics = { multiline = false, diag_source = true },
  lsp = { symbols = { symbol_style = false } },
  files = {
    resume = true,
    actions = {
      ["ctrl-r"] = {
        header = function(o)
          return string.format("switch to '%s'", o.cwd ~= "~" and "~" or "~/.config/nvim")
        end,
        fn = function(_, o)
          require("fzf-lua").resume {
            cwd = o.cwd ~= vim.fn.expand "$HOME" and "~" or "~/.config/nvim",
          }
        end,
      },
    },
  },
  git = {
    status = { preview_pager = false },
    commits = { preview_pager = false },
    bcommits = {
      preview_pager = false,
      actions = {
        ["ctrl-g"] = function(...)
          actions.git_buf_vsplit(...)
          vim.cmd "windo diffthis"
          local switch = vim.api.nvim_replace_termcodes("<C-w>h", true, false, true)
          vim.api.nvim_feedkeys(switch, "t", false)
        end,
      },
    },
  },
  grep = {
    rg_glob = true,
    glob_flag = "--iglob",
    glob_separator = "%s%-%-",
    input_prompt = "Grep> ",
    resume = true,
    actions = {
      ["ctrl-i"] = { actions.toggle_ignore },
    },
  },
  keymap = {
    fzf = {
      ["ctrl-a"] = "select-all",
      ["ctrl-z"] = "deselect-all",
      ["ctrl-t"] = "toggle-all",
    },
  },
  winopts = {
    width = 1,
    height = 0.20,
    row = 1,
    border = "single",
    preview = {
      default = "cat",
      hidden = "hidden",
      border = "border",
      title = false,
      layout = "horizontal",
      horizontal = "right:50%",
    },
  },
  hls = {
    header_bind = "Normal",
    header_text = "Statement",
    buf_name = "Normal",
    buf_nr = "Normal",
    buf_linenr = "Normal",
    buf_flag_cur = "Normal",
    buf_flag_alt = "Normal",
    path_linenr = "Normal",
    path_colnr = "Normal",
    tab_title = "Normal",
    tab_marker = "Normal",
  },
  fzf_opts = {
    ["--ansi"] = "",
    ["--info"] = "inline",
    ["--height"] = "100%",
    ["--layout"] = "reverse",
    ["--border"] = "none",
    ["--color"] = fzf_theme,
  },
  actions = {
    files = {
      ["enter"] = actions.file_edit_or_qf,
      ["ctrl-s"] = actions.file_split,
      ["ctrl-v"] = actions.file_vsplit,
      ["ctrl-t"] = actions.file_tabedit,
      ["ctrl-q"] = actions.file_sel_to_qf,
    },
    grep = {
      ["enter"] = actions.file_edit_or_qf,
      ["ctrl-s"] = actions.file_split,
      ["ctrl-v"] = actions.file_vsplit,
      ["ctrl-t"] = actions.file_tabedit,
      ["ctrl-q"] = actions.file_sel_to_qf,
    },
  },
}
require("fzf-lua").register_ui_select()
-- }}}

-- vimtex {{{2
vim.g.vimtex_parser_bib_backend = "lua"
vim.g.vimtex_view_method = "skim"
vim.g.vimtex_fold_manual = true
vim.g.vimtex_imaps_leader = "\\"
vim.g.vimtex_imaps_enabled = true
-- vim.g.vimtex_syntax_enabled = false
-- }}}

-- slimv {{{2
vim.g.slimv_swank_cmd = ":tabnew term://ros run --load ~/.config/nvim/pack/plugins/opt/slimv/slime/start-swank.lisp"
vim.g.slimv_swank_clojure = ":tabnew term://lein swank"
vim.g.slimv_repl_split = 2
vim.g.slimv_repl_split_size = 50
vim.g.slimv_impl = "sbcl"
vim.g.slimv_lisp = "ros run"
vim.g.slimv_ctags = "ctags"
vim.g.slimv_keybindings = 2
vim.g.slimv_disable_scheme = 1
vim.g.slimv_disable_clojure = 1
vim.g.slimv_unmap_tab = 1
vim.g.slimv_menu = 0
vim.g.paredit_mode = 0
-- }}}
-- vim: set fdm=marker:
-- }}}
