local default_bibfiles = { "~/Documents/notes/references.bib" }
local cachedir = vim.fn.stdpath "state" .. "/fzf-bibtex/"

local pandoc = function(selected, opts)
  local result = vim.fn.system("bibtex-cite", selected)
  vim.api.nvim_put({ result }, "c", false, true)
  if opts.fzf_bibtex.mode == "i" then
    vim.api.nvim_feedkeys("i", "n", true)
  end
end

local cite = function(selected, opts)
  local result = vim.fn.system('bibtex-cite -prefix="\\cite{" -postfix="}" -separator=","', selected)
  vim.api.nvim_put({ result }, "c", false, true)
  if opts.fzf_bibtex.mode == "i" then
    vim.api.nvim_feedkeys("i", "n", true)
  end
end

local markdown_print = function(selected, opts)
  local result = vim.fn.system("bibtex-markdown -cache=" .. cachedir .. " " .. table.concat(vim.b.bibfiles, " "),
    selected)
  local result_lines = {}
  for line in result:gmatch "[^\n]+" do
    table.insert(result_lines, line)
  end
  vim.api.nvim_put(result_lines, "l", true, true)
  if opts.fzf_bibtex.mode == "i" then
    vim.api.nvim_feedkeys("i", "n", true)
  end
end

local file_open = function(selected, opts)
  local result = vim.fn.system("bibtex-cite", selected)
  fh = "/home/chl/Documents/papers/" .. result:sub(2) .. ".pdf"
  vim.api.nvim_command("silent !xdg-open " .. fh .. " &")
end

local fzf_bibtex_menu = function(mode)
  return function()
    if vim.fn.isdirectory(cachedir) == 0 then
      vim.fn.mkdir(cachedir, "p")
    end

    require("fzf-lua").config.set_action_helpstr(pandoc, "@-pandoc")
    require("fzf-lua").config.set_action_helpstr(cite, "\\cite{}")
    require("fzf-lua").config.set_action_helpstr(markdown_print, "markdown-pretty-print")
    require("fzf-lua").config.set_action_helpstr(file_open, "open")

    local filenames = {}
    for i, fullpath in ipairs(vim.b.bibfiles) do
      filenames[i] = vim.fn.fnamemodify(fullpath, ":t")
    end
    local header = table.concat(filenames, "\\ ")

    local default_action = nil
    if vim.bo.ft == "markdown" then
      default_action = pandoc
    elseif vim.bo.ft == "tex" then
      default_action = cite
    end

    return require("fzf-lua").fzf_exec("bibtex-ls " .. "-cache=" .. cachedir .. " " .. table.concat(vim.b.bibfiles, " "),
      {
        actions = {
          ["default"] = default_action,
          ["ctrl-i"] = pandoc,
          ["ctrl-t"] = cite,
          ["ctrl-m"] = markdown_print,
          ["ctrl-o"] = file_open,
        },
        fzf_bibtex = { ["mode"] = mode },
        fzf_opts = { ["--prompt"] = "BibTeX> ", ["--header"] = header },
      })
  end
end

vim.api.nvim_create_autocmd("Filetype", {
  desc = "Set up keymaps for fzf-bibtex",
  group = vim.api.nvim_create_augroup("fzf-bibtex", { clear = true }),
  pattern = { "markdown", "tex", "text", "org", "quarto" },
  callback = function()
    vim.b.bibfiles = default_bibfiles
    vim.keymap.set("n", "grr", fzf_bibtex_menu "n", { buffer = true, desc = "FZF: BibTeX [C]itations" })
    vim.keymap.set("i", "@@", fzf_bibtex_menu "i", { buffer = true, desc = "FZF: BibTeX [C]itations" })
  end,
})
